/* MazeGenerator - Create a maze with depth-first search

Copyright (C) 2012 Michael Billington <michael.billington@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
import java.util.*;

public class Maze {
	/* When generating text output */
	char txtBlank = ' ';
	char txtBlock = '\u2588';

	/* Describing the maze itself */
	Block grid[][];
	int width;
	int height;

	/* Used for weighting the tree-spanning algorithms */
	Random rand = new Random();

	public Maze(int new_width, int new_height) {
		/* Maze constuctor */
		height = new_height;
		width = new_width;
		this.grid = new Block[width][height];

		/* Initialise everything */
		int x;
		int y;
		for(y = 0; y < height; y++) {
			for(x = 0; x < width; x++) {
				this.grid[x][y] = new Block();
				/* Un-join right and bottom side */
				if(y == height - 1) {
					this.grid[x][y].join_bottom = false;
				}
				if(x == width - 1) {
					this.grid[x][y].join_right = false;
				}
			}
		}
	}

	public void toText() {
		/* Output as text (for terminal etc) */
		int x;
		int y;
		int outp_width = width * 2 + 1;
		int outp_height = height * 2 + 1;
		char outp[][] = new char[outp_width][outp_height];

		/* Initialise to all walls */
		for(y = 0; y < outp_height; y++) {
			for(x = 0; x < outp_width; x++) {
				outp[x][y] = txtBlock;
			}
		}
		outp[0][1] = ' ';
		outp[outp_width - 1][outp_height - 2] = ' ';

		/* Renders appropriate gaps in the maze */
		for(y = 0; y < height; y++) {
			for(x = 0; x < width; x++) {
				outp[x * 2 + 1][y * 2 + 1] = ' ';
				if(grid[x][y].join_bottom == true) {
					outp[x * 2 + 1][y * 2 + 2] = ' ';
				}
				if(grid[x][y].join_right == true) {
					outp[x * 2 + 2][y * 2 + 1] = ' ';
				}
			}
		}


		/* Output everything */
		for(y = 0; y < outp_height; y++) {
			for(x = 0; x < outp_width; x++) {
				System.out.print(outp[x][y]);
			}
			System.out.print('\n');
		}
	}

	public void toHTML() {
		/* Generate HTML table from maze */
		int x;
		int y;
		String style;
		System.out.println("<style type=\"text/css\">");
		System.out.println("td { height: 1em; width: 1em; }");
		System.out.println("</style>");
		System.out.println("<table style=\"border-collapse:collapse;\">");
		for(y = 0; y < height; y++) {
			System.out.println("\t<tr>");
			for(x = 0; x < width; x++) {
				style = "";
				if(x == 0 && y != 0) {
					style = style + " border-left: 1px solid #000;";
				}
				if(y == 0) {
					style = style + " border-top: 1px solid #000;";
				}
				if(grid[x][y].join_bottom == false) {
					style = style + " border-bottom: 1px solid #000;";
				}
				if(grid[x][y].join_right == false && !(y == height - 1 && x == width - 1)) {
					style = style + " border-right: 1px solid #000;";
				}
				style = style.trim();
				if(style.equals("")) {
					System.out.println("\t\t<td></td>");
				} else {
					System.out.println("\t\t<td style=\"" + style + "\"></td>");
				}
			}
			System.out.println("\t</tr>");
		}
		System.out.println("</table>");
	}

	public boolean generateDepthFirstSearch() {
		/* Chooses a random point as the root of the tree */
		int x = randomInt(0, width - 1);
		int y = randomInt(0, height - 1);
		followUp(x, y, x, y);	
		return true;
	}

	private boolean followUp(int x, int y, int from_x, int from_y) {
		/* Recursive function to span tree. (Depth-first search) */
		if(x < 0 || y < 0 || x == width || y == height) {
			/* Cancel without causing subsequent operations on this node */
			return false;
		}
		grid[x][y].visited = true;
		int toDo[] = randomise(new int[] {0, 1, 2, 3});
		int i;
		for(i = 0; i < toDo.length; i++) {
			switch(toDo[i]) {
			case 0:
				if(y + 1 != height) {
					if(grid[x][y + 1].visited == false) {
						followUp (x, y + 1, x, y);
					} else if(from_x != x || from_y != y + 1) {
						grid[x][y].join_bottom = false;
					}
				}
				break;
			case 1:
				if(x + 1 != width) {
					if(grid[x + 1][y].visited == false) {
						followUp (x + 1, y, x, y);
					} else if(from_x != x + 1 || from_y != y) {
						grid[x][y].join_right = false;
					}
				}
				break;
			case 2:
				if(x - 1 != -1) {
					if(grid[x - 1][y].visited == false) {
						followUp (x - 1, y, x, y);
					} else if(from_x != x - 1 || from_y != y) {
						grid[x-1][y].join_right = false;
					}
				}
				break;
			case 3:
				if(y - 1 != -1) {
					if(grid[x][y - 1].visited == false) {
						followUp (x, y - 1, x, y);
					} else if(from_x != x || from_y != y - 1) {
						grid[x][y-1].join_bottom = false;
					}
				}
				break;
			}
		}
		return true;
	}

	private int randomInt(int min, int max) {
		/* Random integer between min and max inclusive */
		return rand.nextInt(max - min + 1) + min;
	}

	private int[] randomise(int[] items) {
		/* Un-sort an array of positive integers */
		/* Only use for short arrays! */
		int[] outp = new int[items.length];
		int outp_count = 0;
		int i;
		while(outp_count < items.length) {
			i = randomInt(0, items.length - 1);
			if(items[i] != -1) {
				outp[outp_count] = items[i];
				items[i] = -1;
				outp_count++;
			}
		}
		return outp;
	}

	public void generatePrim() {
		/* Placeholder -- it would be good to implement Prim's
			algorithm, as Java does not like the deep recursion in DFS. */
		int x = randomInt(0, width - 1);
		int y = randomInt(0, height - 1);
		
		
	}
}
